//$$ newmatex.cpp                    NM10Exception handler

// Copyright (C) 1992,3,4,7: R B Davies

#define WANT_STREAM                  // include.h will get stream fns

#include "include.h"                 // include standard files
#include "newmat.h"

#ifdef use_namespace
namespace NEWMAT {
#endif

unsigned long OverflowNM10Exception::Select;
unsigned long SingularNM10Exception::Select;
unsigned long NPDNM10Exception::Select;
unsigned long ConvergenceNM10Exception::Select;
unsigned long ProgramNM10Exception::Select;
unsigned long IndexNM10Exception::Select;
unsigned long VectorNM10Exception::Select;
unsigned long NotSquareNM10Exception::Select;
unsigned long SubMatrixDimensionNM10Exception::Select;
unsigned long IncompatibleDimensionsNM10Exception::Select;
unsigned long NotDefinedNM10Exception::Select;
unsigned long CannotBuildNM10Exception::Select;
unsigned long InternalNM10Exception::Select;



static void MatrixDetails(const GeneralMatrix& A)
// write matrix details to NM10Exception buffer
{
   MatrixBandWidth bw = A.BandWidth(); int ubw = bw.upper; int lbw = bw.lower;
   NM10Exception::AddMessage("MatrixType = ");
   NM10Exception::AddMessage(A.Type().Value());
   NM10Exception::AddMessage("  # Rows = "); NM10Exception::AddInt(A.Nrows());
   NM10Exception::AddMessage("; # Cols = "); NM10Exception::AddInt(A.Ncols());
   if (lbw >=0)
      { NM10Exception::AddMessage("; lower BW = "); NM10Exception::AddInt(lbw); }
   if (ubw >=0)
      { NM10Exception::AddMessage("; upper BW = "); NM10Exception::AddInt(ubw); }
   NM10Exception::AddMessage("\n");
}

NPDNM10Exception::NPDNM10Exception(const GeneralMatrix& A)
   : Runtime_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: matrix not positive definite\n\n");
   MatrixDetails(A);
   Tracer::AddTrace();
}

SingularNM10Exception::SingularNM10Exception(const GeneralMatrix& A)
   : Runtime_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: matrix is singular\n\n");
   MatrixDetails(A);
   Tracer::AddTrace();
}

ConvergenceNM10Exception::ConvergenceNM10Exception(const GeneralMatrix& A)
   : Runtime_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: process fails to converge\n\n");
   MatrixDetails(A);
   Tracer::AddTrace();
}

ConvergenceNM10Exception::ConvergenceNM10Exception(const char* c) : Runtime_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: ");
   AddMessage(c); AddMessage("\n\n");
   if (c) Tracer::AddTrace();
}

OverflowNM10Exception::OverflowNM10Exception(const char* c) : Runtime_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: ");
   AddMessage(c); AddMessage("\n\n");
   if (c) Tracer::AddTrace();
}

ProgramNM10Exception::ProgramNM10Exception(const char* c) : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: ");
   AddMessage(c); AddMessage("\n\n");
   if (c) Tracer::AddTrace();
}

ProgramNM10Exception::ProgramNM10Exception(const char* c, const GeneralMatrix& A)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: ");
   AddMessage(c); AddMessage("\n\n");
   MatrixDetails(A);
   if (c) Tracer::AddTrace();
}

ProgramNM10Exception::ProgramNM10Exception(const char* c, const GeneralMatrix& A,
   const GeneralMatrix& B) : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: ");
   AddMessage(c); AddMessage("\n\n");
   MatrixDetails(A); MatrixDetails(B);
   if (c) Tracer::AddTrace();
}

ProgramNM10Exception::ProgramNM10Exception(const char* c, MatrixType a, MatrixType b)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: ");
   AddMessage(c); AddMessage("\nMatrixTypes = ");
   AddMessage(a.Value()); AddMessage("; ");
   AddMessage(b.Value()); AddMessage("\n\n");
   if (c) Tracer::AddTrace();
}

VectorNM10Exception::VectorNM10Exception() : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: cannot convert matrix to vector\n\n");
   Tracer::AddTrace();
}

VectorNM10Exception::VectorNM10Exception(const GeneralMatrix& A)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: cannot convert matrix to vector\n\n");
   MatrixDetails(A);
   Tracer::AddTrace();
}

NotSquareNM10Exception::NotSquareNM10Exception(const GeneralMatrix& A)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: matrix is not square\n\n");
   MatrixDetails(A);
   Tracer::AddTrace();
}

SubMatrixDimensionNM10Exception::SubMatrixDimensionNM10Exception()
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: incompatible submatrix dimension\n\n");
   Tracer::AddTrace();
}

IncompatibleDimensionsNM10Exception::IncompatibleDimensionsNM10Exception()
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: incompatible dimensions\n\n");
   Tracer::AddTrace();
}

IncompatibleDimensionsNM10Exception::IncompatibleDimensionsNM10Exception
   (const GeneralMatrix& A, const GeneralMatrix& B)
      : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: incompatible dimensions\n\n");
   MatrixDetails(A); MatrixDetails(B);
   Tracer::AddTrace();
}

NotDefinedNM10Exception::NotDefinedNM10Exception(const char* op, const char* matrix)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: ");
   AddMessage(op);
   AddMessage(" not defined for ");
   AddMessage(matrix);
   AddMessage("\n\n");
   Tracer::AddTrace();
}

CannotBuildNM10Exception::CannotBuildNM10Exception(const char* matrix)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: cannot build matrix type ");
   AddMessage(matrix); AddMessage("\n\n");
   Tracer::AddTrace();
}

IndexNM10Exception::IndexNM10Exception(int i, const GeneralMatrix& A)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: index error: requested index = ");
   AddInt(i); AddMessage("\n\n");
   MatrixDetails(A);
   Tracer::AddTrace();
}

IndexNM10Exception::IndexNM10Exception(int i, int j, const GeneralMatrix& A)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: index error: requested indices = ");
   AddInt(i); AddMessage(", "); AddInt(j);
   AddMessage("\n\n");
   MatrixDetails(A);
   Tracer::AddTrace();
}


IndexNM10Exception::IndexNM10Exception(int i, const GeneralMatrix& A, bool)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("detected by Newmat: element error: requested index (wrt 0) = ");
   AddInt(i);
   AddMessage("\n\n");
   MatrixDetails(A);
   Tracer::AddTrace();
}

IndexNM10Exception::IndexNM10Exception(int i, int j, const GeneralMatrix& A, bool)
   : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage(
      "detected by Newmat: element error: requested indices (wrt 0) = ");
   AddInt(i); AddMessage(", "); AddInt(j);
   AddMessage("\n\n");
   MatrixDetails(A);
   Tracer::AddTrace();
}

InternalNM10Exception::InternalNM10Exception(const char* c) : Logic_error()
{
   Select = NM10Exception::Select;
   AddMessage("internal error detected by Newmat: please inform author\n");
   AddMessage(c); AddMessage("\n\n");
   Tracer::AddTrace();
}




/************************* ExeCounter functions *****************************/

#ifdef DO_REPORT

int ExeCounter::nreports;                      // will be set to zero

ExeCounter::ExeCounter(int xl, int xf) : line(xl), fileid(xf), nexe(0) {}

ExeCounter::~ExeCounter()
{
   nreports++;
   cout << "REPORT  " << setw(6) << nreports << "  "
      << setw(6) << fileid << "  " << setw(6) << line
      << "  " << setw(6) << nexe << "\n";
}

#endif

/**************************** error handler *******************************/

void MatrixErrorNoSpace(void* v) { if (!v) Throw(Bad_alloc()); }
// throw exception if v is null




/************************* miscellanous errors ***************************/


void CroutMatrix::GetRow(MatrixRowCol&)
   { Throw(NotDefinedNM10Exception("GetRow","Crout")); }
void CroutMatrix::GetCol(MatrixRowCol&)
   { Throw(NotDefinedNM10Exception("GetCol","Crout")); }
void CroutMatrix::operator=(const BaseMatrix&)
   { Throw(NotDefinedNM10Exception("=","Crout")); }
void BandLUMatrix::GetRow(MatrixRowCol&)
   { Throw(NotDefinedNM10Exception("GetRow","BandLUMatrix")); }
void BandLUMatrix::GetCol(MatrixRowCol&)
   { Throw(NotDefinedNM10Exception("GetCol","BandLUMatrix")); }
void BandLUMatrix::operator=(const BaseMatrix&)
   { Throw(NotDefinedNM10Exception("=","BandLUMatrix")); }
void BaseMatrix::IEQND() const
   { Throw(NotDefinedNM10Exception("inequalities", "matrices")); }
#ifdef TEMPS_DESTROYED_QUICKLY_R
   ReturnMatrixX::ReturnMatrixX(const ReturnMatrixX& tm)
     : gm(tm.gm) { Throw(ProgramNM10Exception("ReturnMatrixX error")); }
#endif


#ifdef use_namespace
}
#endif

